--                              -*- Mode: Ada -*-
-- Filename        : pixmaps.adb
-- Description     : Pixmaps
-- Author          : COL Ressler
-- Created On      : Sun Mar 10 17:23:35 2002
-- Last Modified By: COL Ressler
-- Last Modified On: .
-- Update Count    : 0
-- Status          : Unknown, Use with caution!
with Ada.Text_IO;
with Ada.Integer_Text_IO;
with Ada.Sequential_IO;
with Ada.Unchecked_Deallocation;
with Win32.GL;
use Win32.GL;
with Win32.GLU;
use Win32.GLU;

package body Pixmaps is

   procedure Free_Pixels is
      new Ada.Unchecked_Deallocation(RGB_Pixel_Array_Type, RGB_Pixel_Array_Ptr_Type);

   procedure Set_OpenGL_Texture(Pixmap : in Pixmap_Type; Identifier : in Integer) is
      Dummy_Rtn_Val : Win32.INT;
   begin
      GlTexParameterI(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
      GlTexParameterI(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
      GlBindTexture(GL_TEXTURE_2D, GLuint(Identifier));
      Dummy_Rtn_Val :=
      GluBuild2DMipmaps(GL_TEXTURE_2D, 3, -- three color components
                        Glint(Pixmap.N_Columns), Glint(Pixmap.N_Rows),
                        GL_RGB, GL_UNSIGNED_BYTE,
                        Pixmap.Pixels(Pixmap.Pixels.all'First(1),
                                      Pixmap.Pixels.all'First(2))'Address);
   end;

   procedure Clear_Pixmap(Pixmap : in out Pixmap_Type) is
   begin
      Free_Pixels(Pixmap.Pixels);
      Pixmap.N_Rows := 0;
      Pixmap.N_Columns := 0;
   end;

   function Height(Pixmap : in Pixmap_Type) return Natural is
   begin
      return Pixmap.N_Rows;
   end;

   function Width(Pixmap : in Pixmap_Type) return Natural is
   begin
      return Pixmap.N_Columns;
   end;

   procedure Get_Size(Pixmap : in Pixmap_Type;
                      Width, Height : out Natural) is
   begin
      Height := Pixmap.N_Rows;
      Width := Pixmap.N_Columns;
   end;

   function Pixel(Pixmap : in Pixmap_Type;
                  Row, Column : in Natural) return RGB_Pixel_Type is
   begin
      return Pixmap.Pixels(Row, Column);
   end;

   procedure Get_Pixel(Pixmap : in Pixmap_Type;
                       Row, Column : in Natural;
                       Pixel : out RGB_Pixel_Type) is
   begin
      Pixel := Pixmap.Pixels(Row, Column);
   end;

   -- Set a pixel value.
   procedure Set_Pixel(Pixmap : in out Pixmap_Type;
                       Row, Column : in Natural;
                       Pixel : in RGB_Pixel_Type) is
   begin
      Pixmap.Pixels(Row, Column) := Pixel;
   end;

   -- XY versions of get and set are for convenience
   -- of those using (x,y) coordinates rather than (row, column).
   -- These functions just call the above with coordinates
   -- reversed.
   function Pixel_XY(Pixmap : in Pixmap_Type;
                     X, Y : in Natural) return RGB_Pixel_Type is
   begin
      return Pixel(Pixmap, Y, X);
   end;

   procedure Get_Pixel_XY(Pixmap : in Pixmap_Type;
                          X, Y : in Natural;
                          Pixel : out RGB_Pixel_Type) is
   begin
      Get_Pixel(Pixmap, Y, X, Pixel);
   end;

   package Sequential_Byte_IO is new Ada.Sequential_IO(Byte_Type);
   use Sequential_Byte_IO;

   type Unsigned_Long_Type is mod 2**32;
   subtype Long_Byte_Count_Range_Type is Natural range 1..4;

   procedure Read_Little_Endian(BMP_File : in out File_Type;
                                Val : out Unsigned_Long_Type;
                                N_Bytes : in Long_Byte_Count_Range_Type) is
      Byte_Val : Byte_Type;
      Multiplier : Unsigned_Long_Type := 1;
   begin
      Val := 0;
      for Byte_Number in 0..N_Bytes - 1 loop
         Read(BMP_File, Byte_Val);
         Val := Val + Unsigned_Long_Type(Byte_Val) * Multiplier;
         Multiplier := Multiplier * (2 ** 8);
      end loop;
   end;

   procedure Write_Little_Endian(BMP_File : in out File_Type;
                                 Val : in Unsigned_Long_Type;
                                 N_Bytes : in Long_Byte_Count_Range_Type) is
      Divisor : Unsigned_Long_Type := 1;
   begin
      for Byte_Number in 0..N_Bytes - 1 loop
         Write(BMP_File, Byte_Type(Val / Divisor));
         Divisor := Divisor * (2 ** 8);
      end loop;
   end;

   -- This two-byte quantity is always first in a BMP file.
   -- Such signature values are conventionally called magic numbers.
   BMP_Magic : constant Unsigned_Long_Type :=
     Character'Pos('B') + 2**8 * Character'Pos('M');

   procedure Read_BMP_File(File_Name : in String;
                           Pixmap : out Pixmap_Type) is

      BMP_File : File_Type;

      -- Items in BMP format specification.
      Magic,
      File_Size,
      Reserved_1,
      Reserved_2,
      Offset,
      Header_Size,
      N_Columns,
      N_Rows,
      N_Planes,
      N_Bits_Per_Pixel,
      Compression,
      Image_Size,
      X_Pels,
      Y_Pels,
      N_LUT_Entries,
      N_Imp_Colors : Unsigned_Long_Type;

      -- Count of total bytes read so far.
      N_Bytes_Read : Unsigned_Long_Type := 0;

      -- Place to read a padding byte.
      Wasted_Byte : Byte_Type;

      -- Check that two unsigned 32 bit quantities match and
      -- raise format exception if not.
      procedure Assert(Actual, Should_Be : in Unsigned_Long_Type) is
      begin
         if Actual /= Should_Be then
            raise BMP_Format_Error;
         end if;
      end Assert;

   begin

      -- Get rid of old image if there was one.
      Clear_Pixmap(Pixmap);

      Open(BMP_File, In_File, File_Name);

      -- Get and check the magic header.
      Read_Little_Endian(BMP_File, Magic, 2);            Assert(Magic, BMP_Magic);

      Read_Little_Endian(BMP_File, File_Size, 4);
      Read_Little_Endian(BMP_File, Reserved_1, 2);       Assert(Reserved_1, 0);
      Read_Little_Endian(BMP_File, Reserved_2, 2);       Assert(Reserved_2, 0);
      Read_Little_Endian(BMP_File, Offset, 4);
      Read_Little_Endian(BMP_File, Header_Size, 4);      Assert(Header_Size, 40);
      Read_Little_Endian(BMP_File, N_Columns, 4);
      Read_Little_Endian(BMP_File, N_Rows, 4);
      Read_Little_Endian(BMP_File, N_Planes, 2);         Assert(N_Planes, 1);
      Read_Little_Endian(BMP_File, N_Bits_Per_Pixel, 2); Assert(N_Bits_Per_Pixel, 24);
      Read_Little_Endian(BMP_File, Compression, 4);      Assert(Compression, 0);
      Read_Little_Endian(BMP_File, Image_Size, 4);
      Read_Little_Endian(BMP_File, X_Pels, 4);
      Read_Little_Endian(BMP_File, Y_Pels, 4);
      Read_Little_Endian(BMP_File, N_LUT_Entries, 4);    Assert(N_LUT_Entries, 0);
      Read_Little_Endian(BMP_File, N_Imp_Colors, 4);     Assert(N_Imp_Colors, 0);

      Pixmap.N_Rows := Natural(N_Rows);
      Pixmap.N_Columns := Natural(N_Columns);
      Pixmap.Pixels := new RGB_Pixel_Array_Type(1..Pixmap.N_Rows, 1..Pixmap.N_Columns);

      for Row in 1..Pixmap.N_Rows loop

         for Column in 1..Pixmap.N_Columns loop
            declare
               Current_Pixel : RGB_Pixel_Type Renames Pixmap.Pixels.all(Row, Column);
            begin
               Read(BMP_File, Current_Pixel.Red);
               Read(BMP_File, Current_Pixel.Green);
               Read(BMP_File, Current_Pixel.Blue);
            end;
            N_Bytes_Read := N_Bytes_Read + 3;
         end loop;

         -- Slurp rest of row.  BMP file rows are always multiples of 4 bytes.
         while N_Bytes_Read mod 4 /= 0 loop
            Read(BMP_File, Wasted_Byte);
            N_Bytes_Read := N_Bytes_Read + 1;
         end loop;

      end loop;

      Assert(Image_Size, N_Bytes_Read);

      Close(BMP_File);

   exception
      when Name_Error =>
         raise;
      when others =>
         raise BMP_Format_Error;
   end;

   procedure Write_BMP_File(File_Name : in String;
                            Pixmap : in Pixmap_Type) is

      BMP_File : File_Type;

      -- Computed header values.
      Bytes_Per_Row,
      Image_Size,
      File_Size : Unsigned_Long_Type;

      -- Constant header values.
      Reserved_1 : constant Unsigned_Long_Type := 0;
      Reserved_2 : constant Unsigned_Long_Type := 0;
      Offset : constant Unsigned_Long_Type := 54;
      Header_Size : constant Unsigned_Long_Type := 40;
      N_Rows : constant Unsigned_Long_Type := Unsigned_Long_Type(Pixmap.N_Rows);
      N_Columns : constant Unsigned_Long_Type := Unsigned_Long_Type(Pixmap.N_Columns);
      N_Planes : constant Unsigned_Long_Type := 1;
      N_Bits_Per_Pixel : constant Unsigned_Long_Type := 24;
      Compression : constant Unsigned_Long_Type := 0;
      X_Pels : constant Unsigned_Long_Type := 0;
      Y_Pels : constant Unsigned_Long_Type := 0;
      N_LUT_Entries : constant Unsigned_Long_Type := 0;
      N_Imp_Colors  : constant Unsigned_Long_Type := 0;

      N_Bytes_Written : Natural := 0;
   begin

      -- Compute bytes in a row with 3 bytes per pixel.
      Bytes_Per_Row := N_Columns * 3;

      -- Add in padding if the result is not an even multiple of 4.
      if Bytes_Per_Row mod 4 /= 0 then
         Bytes_Per_Row := Bytes_Per_Row + (4 - Bytes_Per_Row mod 4);
      end if;

      -- Compute image size and total file size.
      Image_Size := Bytes_Per_Row * N_Rows;
      File_Size := Offset + Image_Size;

      Create(BMP_File, Out_File, File_Name);

      -- Get and check the magic header.
      Write_Little_Endian(BMP_File, BMP_Magic, 2);

      Write_Little_Endian(BMP_File, File_Size, 4);
      Write_Little_Endian(BMP_File, Reserved_1, 2);
      Write_Little_Endian(BMP_File, Reserved_2, 2);
      Write_Little_Endian(BMP_File, Offset, 4);
      Write_Little_Endian(BMP_File, Header_Size, 4);
      Write_Little_Endian(BMP_File, N_Columns, 4);
      Write_Little_Endian(BMP_File, N_Rows, 4);
      Write_Little_Endian(BMP_File, N_Planes, 2);
      Write_Little_Endian(BMP_File, N_Bits_Per_Pixel, 2);
      Write_Little_Endian(BMP_File, Compression, 4);
      Write_Little_Endian(BMP_File, Image_Size, 4);
      Write_Little_Endian(BMP_File, X_Pels, 4);
      Write_Little_Endian(BMP_File, Y_Pels, 4);
      Write_Little_Endian(BMP_File, N_LUT_Entries, 4);
      Write_Little_Endian(BMP_File, N_Imp_Colors, 4);

      for Row in 1..Pixmap.N_Rows loop

         for Column in 1..Pixmap.N_Columns loop
            declare
               Current_Pixel : RGB_Pixel_Type Renames Pixmap.Pixels.all(Row, Column);
            begin
               Write(BMP_File, Current_Pixel.Blue);
               Write(BMP_File, Current_Pixel.Green);
               Write(BMP_File, Current_Pixel.Red);
            end;
            N_Bytes_Written := N_Bytes_Written + 3;
         end loop;

         -- Slurp rest of row.  BMP file rows are always multiples of 4 bytes.
         while N_Bytes_Written mod 4 /= 0 loop
            Write(BMP_File, 0);
            N_Bytes_Written := N_Bytes_Written + 1;
         end loop;

      end loop;

      Close(BMP_File);
   end;

   -- Managers to keep track of pixel array when pixmap is assigned or
   -- deleted (going out of scope or deallocated).
   procedure Adjust(Pixmap: in out Pixmap_Type) is
   begin
      Pixmap.Pixels := new RGB_Pixel_Array_Type'(Pixmap.Pixels.all);
   end;

   procedure Finalize(Pixmap : in out Pixmap_Type) is
   begin
      Free_Pixels(Pixmap.Pixels);
   end;

end Pixmaps;
